/**
  @file             SM_LLL.h
  @version          0.1.0.0

  @brief            
  @details          

  Project           Stepper Motor Driver
  Platform          MPC5606S

  SWVersion         0.1.0


  (c) Copyright 2011 Freescale Semiconductor Inc.
  All Rights Reserved.
*/
/*==================================================================================================
==================================================================================================*/

#ifndef SM_LLL
    #define SM_LLL

/*==================================================================================================
                                       DEFINES AND MACROS
==================================================================================================*/

/* Selection between two lengths of the Sin/Cos table */
#ifdef SM_TABLE_128
	#define LLL_TABLELENGTH 			128u		/* SM table of 128 elements is selected */
    #endif

#ifdef SM_TABLE_256
	#define LLL_TABLELENGTH 			256u		/* SM table of 256 elements is selected */ 
    #endif

/* Number of table elements corresponding to 90 degrees (1 full step) */
#define LLL_DEG_90 						(LLL_TABLELENGTH / 4u);	
/* Number of table elements corresponding to 45 degrees (1/2 full step) */	 
#define LLL_DEG_45 						(LLL_TABLELENGTH / 8u);	

/* STM_0 Timer */
#define LLL_TIMER0_VALUE(x)				RAL_RegRead(&RAL_STM0_REG_CMP(x))
#define LLL_TIMER0_INT_CLEAR(x)			RAL_RegBitSet(&RAL_STM0_REG_CIR(x),RAL_STM0_BIT_CIR_CIF)

/* STM_1 Timer */
#define LLL_TIMER1_VALUE(x)				RAL_RegRead(&RAL_STM1_REG_CMP(x))
#define LLL_TIMER1_INT_CLEAR(x)			RAL_RegBitSet(&RAL_STM1_REG_CIR(x),RAL_STM1_BIT_CIR_CIF)

/* Read from the corresponding SSD Integration Accumulator Register (x - SSD module index) */  
#define LLL_SSDX_ITGACC(x)				RAL_RegRead(&(x)->RAL_SSDX_REG_ITGACC)

/* Clear SSD Integration Expired Interrupt Flag in the corresponding SSD module (x - SSD module index) */
#define LLL_SSDX_INT_CLEAR(x)			RAL_RegBitSet(&(x)->RAL_SSDX_REG_IRQ,RAL_SSDX_BIT_IRQ_ITGIF)

/* Read actual "STEP" value from CONTROL register of the corresponding SSD module (x - SSD module index) */ 
#define LLL_SSDX_ACTUAL_STEP(x)			(((RAL_RegRead(&(x)->RAL_SSDX_REG_CONTROL)) >> 13u) & 0x03u)
			
/* Increment actual "STEP" value in CONTROL register of the corresponding SSD module (x - SSD module index) */  			
#define LLL_SSDX_STEP_INC(x)			(x)->RAL_SSDX_CONTROL_STEP++										

/* Decrement actual "STEP" value in CONTROL register of the corresponding SSD module (x - SSD module index) */
#define LLL_SSDX_STEP_DEC(x)			(x)->RAL_SSDX_CONTROL_STEP--										

/* Set Trigger bit in CONTROL register of the corresponding SSD module (x - SSD module index) */							
#define LLL_SSDX_BIS_TRIG(x)			RAL_RegBitSet(&(x)->RAL_SSDX_REG_CONTROL,RAL_SSDX_BIT_CONTROL_TRIG)											
		
/* Enable SSD Integration Expired Interrupt of the corresponding SSD module (x - SSD module index) */ 
#define LLL_SSDX_INT_ITG_ENABLE(x)		RAL_RegBitSet(&(x)->RAL_SSDX_REG_IRQ,RAL_SSDX_BIT_IRQ_ITGIE)

/* Disable SSD Integration Expired Interrupt of the corresponding SSD module (x - SSD module index) */	
#define LLL_SSDX_INT_ITG_DISABLE(x)		RAL_RegBitClr(&(x)->RAL_SSDX_REG_IRQ,RAL_SSDX_BIT_IRQ_ITGIE)	
										
/* Enable SMC Timer Counter Overflow Interrupt (common for all the motors) */ 
#define LLL_SMC_OVF_INT_ENABLE			RAL_RegWrite(0x01u,&RAL_SMC_REG_MCCTL1)

/* Disable SMC Timer Counter Overflow Interrupt (common for all the motors) */
#define LLL_SMC_OVF_INT_DISABLE			RAL_RegWrite(0x00u,&RAL_SMC_REG_MCCTL1)

/* Clear SMC Timer Counter Overflow Interrupt flag (common for all the motors) */
#define LLL_SMC_OVF_INT_CLEAR			RAL_RegBitSet(&RAL_SMC_REG_MCCTL0,RAL_SMC_BIT_MCCTL0_MCTOIF)

/* SIU configuration */
#define LLL_SIUL2_COIL_COS_A				3u
#define LLL_SIUL2_COIL_COS_B				2u
#define LLL_SIUL2_COIL_SIN_A				1u
#define LLL_SIUL2_COIL_SIN_B				0u

#define LLL_SIUL2_SMC_OFFSET				32u

#define LLL_SIUL2_MOTOR_0_OFFSET			48u
#define LLL_SIUL2_MOTOR_1_OFFSET			52u
#define LLL_SIUL2_MOTOR_2_OFFSET			32u
#define LLL_SIUL2_MOTOR_3_OFFSET			36u
#define LLL_SIUL2_MOTOR_4_OFFSET			40u
#define LLL_SIUL2_MOTOR_5_OFFSET			44u

#define LLL_SIUL2_MOTOR(x)					(((x) * 4) + LLL_SIUL2_SMC_OFFSET)

#define LLL_SIUL2_GPIO						0u
#define LLL_SIUL2_SMC						0x00000001u
#define LLL_SIUL2_SSD						0x00000002u


#define LLL_SIUL2_OUTPUT(ch,val)			RAL_RegWrite((val),&SIUL2.MSCR[((ch) + LLL_SIUL2_COIL_COS_A)].R);\
											RAL_RegWrite((val),&SIUL2.MSCR[((ch) + LLL_SIUL2_COIL_COS_B)].R);\
											RAL_RegWrite((val),&SIUL2.MSCR[((ch) + LLL_SIUL2_COIL_SIN_A)].R);\
											RAL_RegWrite((val),&SIUL2.MSCR[((ch) + LLL_SIUL2_COIL_SIN_B)].R)
										
/*==================================================================================================
                                 GLOBAL VARIABLE DECLARATIONS
==================================================================================================*/

/* Function initializes HW modules (SMC, SIU, timer) of the corresponding motor - will be modified for the HALO */
void SM_LLL_InitHwModules(SM_MotorParams_t *prMtr);

/* Function Initializes SSD module of the corresponding motor */
void SM_LLL_SsdInit(SM_MotorParams_t *prMtr);

/* Function disables HW modules of the corresponding motor */ 
void SM_LLL_DisableHwModules(SM_MotorParams_t *prMtr);

/* Function drives PWM channels of the corresponding motor */
void SM_LLL_DrivePwms(SM_MotorParams_t *prMtr);

/* Function operates RTZ movement */
void SM_LLL_Rtz(SM_MotorParams_t *prMtr);

/* Function handles SSD interrupt */
void SM_LLL_SsdInterrupt(SM_MotorParams_t *prMtr);

/* Function initializes algorithm timer - will be modified for the HALO */
void SM_LLL_TimerInit(SM_MotorId_t eMtrId);

/* Function disables algorithm timer - will be modified for the HALO */
void SM_LLL_TimerDisable(SM_MotorId_t eMtrId);

#endif /* SM_LLL */